"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import Icon from "@/components/icons/Icon";
import { useLanguage } from "@/contexts/LanguageContext";

interface PostProps {
  params: Promise<{ slug: string }>; // ✅ params is a Promise now
}

export default function Post({ params }: PostProps) {
  const { slug } = use(params); // ✅ unwrap Promise
  const { language } = useLanguage();
  const [post, setPost] = useState<any | null | undefined>(null);

  useEffect(() => {
    const fetchPost = async () => {
      try {
        const data = await wordpressAPI.getPostBySlug(slug, language);
        setPost(data || undefined);
      } catch (error) {
        console.error("Error loading post:", error);
        setPost(undefined);
      }
    };
    fetchPost();
  }, [slug, language]);

  if (post === null) {
    // Show skeleton loader while loading
    return (
      <div className="te-post">
        <div className="container mx-auto max-w-5xl my-16">
          <div className="mb-6 text-center space-y-3">
            <div className="skeleton w-3/4 h-10 mx-auto rounded-lg"></div>
            <div className="skeleton w-1/2 h-6 mx-auto rounded-md mt-4"></div>
            <div className="flex justify-center gap-4 text-gray-600 dark:text-gray-300 mb-2">
              <div className="skeleton w-16 h-6 rounded-md"></div>
              <div className="skeleton w-16 h-6 rounded-md"></div>
              <div className="skeleton w-16 h-6 rounded-md"></div>
            </div>
          </div>
          <div className="mb-6 relative w-full">
            <div className="aspect-[16/9] object-cover">
              <div className="skeleton w-full h-64 rounded-xl"></div>
            </div>
          </div>
          <div className="prose prose-md dark:prose-invert max-w-none">
            <div className="skeleton w-full h-40 rounded-md mb-4"></div>
            <div className="skeleton w-full h-40 rounded-md mb-4"></div>
          </div>
        </div>
      </div>
    );
  }

  if (post === undefined) notFound(); // not found

  // Extract post image (featured image with the new size)
  const featuredImage =
    post._embedded?.["wp:featuredmedia"]?.[0]?.media_details?.sizes["single-post-featured-image"]?.source_url ||
    post._embedded?.["wp:featuredmedia"]?.[0]?.source_url;

  // Extract post metadata
  const author = post._embedded?.author?.[0]?.name || "Unknown Author";
  const date = post.date ? new Date(post.date).toLocaleDateString() : "Unknown Date";
  const categories = post._embedded?.["wp:term"]?.[0]?.map((cat: any) => cat.name).join(", ") || "No Categories";

  return (
    <div className="te-post">
      <div className="container mx-auto max-w-5xl my-16">
        <div className="mb-6 text-center space-y-3">
          <h1 className="text-3xl md:text-4xl font-bold dark:text-white"
            dangerouslySetInnerHTML={{
              __html: typeof post.title === "string" ? post.title : post.title?.rendered || "",
            }}
          />

          <div className="text-sm flex justify-center gap-4 text-gray-600 dark:text-gray-300 mb-2">
            <span className="inline-flex items-center gap-1">
              <Icon name="author" size={16} /> {author}
            </span>

            <span className="inline-flex items-center gap-1">
              <Icon name="date" size={16} />
              {date}
            </span>

            <span className="inline-flex items-center gap-1">
              <Icon name="categories" size={16} />
              {categories}
            </span>

          </div>
        </div>

        {featuredImage && (
          <div className="mb-6 relative w-full">
            <div className="aspect-[16/9] object-cover">
              <img
                src={featuredImage}
                alt={post.title?.rendered || "Featured image"}
                className="rounded-xl object-cover w-full h-full"
              />
            </div>
          </div>
        )}

        <div className="prose prose-md dark:prose-invert prose-img:rounded-xl prose-a:text-primary-600 dark:prose-a:text-primary-400 hover:prose-a:text-gray-500 max-w-none"
          dangerouslySetInnerHTML={{
            __html: typeof post.content === "string" ? post.content : post.content?.rendered || "",
          }}
        />

      </div>
    </div>
  );
}
